/*
 * Copyright (c) 2009 Marcus Glocker <marcus@nazgul.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <poll.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <termios.h>
#include <time.h>
#include <unistd.h>

#include "../libmy/file.h"
#include "../apc.h"

/*
 * Prototypes
 */
void	signal_handler(int);
void	usage(void);
void	poll_init(struct pollfd *, int);
void	poll_add(struct pollfd *, int);
void	poll_remove(struct pollfd *, int);
int	ups_write(int);
int	ups_read(char *);
int	ups_query(struct ups_query *);
void	ups_startup_test(void);
void	ups_check_condition(void);
void	alert_online(void);
void	alert_battery(void);
void	alert_battery_low(void);
void	alert_battery_crit(void);
void	alert_shutdown(void);
void	trap_run(char *);
int	client_request(int);
int	load_config(int);
void	tell(int, char *, ...);
void	debug(char *, ...);

/*
 * Global variables
 */
extern char		*__progname;
char			 cfgfile[128];
int			 optdebug;
int			 optcfgfile;
int			 fdserial;
volatile sig_atomic_t	 quit = 0;
struct apcd_config	 cfg;
struct ups_query	 query;
struct ups_query	 querylast;

void
signal_handler(int sig)
{
	switch (sig) {
	case SIGALRM:
		ups_query(&query);
		ups_check_condition();
		break;
	case SIGCHLD: {
		int status;
		while (waitpid(-1, &status, WNOHANG) > 0);
		break;
	}
	case SIGHUP:
		if (load_config(1) == 0)
			tell(LOG_INFO, "config reloaded");
		else
			tell(LOG_INFO, "config not reloaded");
		break;
	case SIGTERM:
		quit = 1;
		break;
	default:
		tell(LOG_INFO, "unhandled signal");
		break;
	}
}

void
usage(void)
{
	fprintf(stderr, "usage: %s ", __progname);
	fprintf(stderr, "[-dhv] [-c configfile]\n");
	exit(1);
}

int
main(int argc, char *argv[])
{
	int sdlistener;
	int sdclient;
	int size;
	int r;
	int n;
	int i;
	int nfds;
	int numclients;
	int fcntlorig;
	int ch;
	FILE *f;
	pid_t pid;
	struct sockaddr_in sa;
	struct sockaddr_in ca;
	socklen_t sdsize;
	struct pollfd fdpoll[APCD_POLL_FDS];
	struct itimerval t;
	struct termios term;

	pid = sdlistener = 0;

	while ((ch = getopt(argc, argv, "dhvc:")) != -1) {
		switch (ch) {
		case 'd':
			optdebug = 1;
			break;
		case 'h':
			usage();
			/* NOTREACHED */
		case 'v':
			printf("%s 0.2\n", __progname);
			return (0);
			break;
		case 'c':
			optcfgfile = 1;
			strlcpy(cfgfile, optarg, sizeof(cfgfile));
			break;
		default:
			usage();
			/* NOTREACHED */
		}
	}
	if (!optcfgfile)
		strlcpy(cfgfile, APCD_CONFIG, sizeof(cfgfile));

	/*
	 * Parse configuration file.
	 */
	r = load_config(0);
	if (r == -1)
		return (1);

	/*
	 * Establish serial connection to UPS
	 */
	fdserial = open(cfg.device, O_RDWR | O_NONBLOCK);
	if (fdserial == -1)
		err(1, "open");

	term.c_cflag = CS8 | CRTSCTS | CREAD;
	term.c_iflag = IGNBRK | IGNPAR;
	term.c_oflag = 0;
	term.c_lflag = 0;
	term.c_cc[VMIN] = 1;
	term.c_cc[VTIME] = 50;
	cfsetspeed(&term, 2400);
	r = tcsetattr(fdserial, TCSADRAIN, &term);
	if (r == -1)
		err(1, "tcsetattr");

	ioctl(fdserial, TIOCSDTR, 0);
	ioctl(fdserial, TIOCSCTTY, 0);

	fcntlorig = fcntl(fdserial, F_GETFL, 0);
	if (fcntlorig == -1)
		err(1, "fcntl");
	fcntl(fdserial, F_SETFL, fcntlorig & ~O_NONBLOCK);

	/*
	 * Startup network listener for client connections
	 */
	if (cfg.listener != -1) {
		sdlistener = socket(AF_INET, SOCK_STREAM, 0);
		if (sdlistener == -1)
			err(1, "socket");

		r = fcntl(sdlistener, F_SETFL,
		    fcntl(sdlistener, F_GETFL) | O_NONBLOCK);
		if (r == -1)
			err(1, "fcntl");

		size = 1;
		r = setsockopt(sdlistener, SOL_SOCKET, SO_REUSEADDR,
		    (char *)&size, sizeof(size));
		if (r == -1)
			err(1, "setsockopt");

		sa.sin_family = AF_INET;
		sa.sin_port = htons(cfg.listener);
		sa.sin_addr.s_addr = INADDR_ANY;
		memset(&sa.sin_zero, 0, sizeof(sa.sin_zero));
		r = bind(sdlistener, (struct sockaddr *)&sa, sizeof(sa));
		if (r == -1)
			err(1, "bind");

		r = listen(sdlistener, APCD_LISTEN_CON);
		if (r == -1)
			err(1, "listen");
	}

	/*
	 * Setup signal handler.
	 */
	signal(SIGALRM, signal_handler);
	signal(SIGCHLD, signal_handler);
	signal(SIGHUP, signal_handler);
	signal(SIGTERM, signal_handler);

	/*
	 * Open syslog.
	 */
	openlog("apcd", LOG_PID | LOG_NDELAY, LOG_DAEMON);

	/*
	 * Daemonize.
	 */
	if (!optdebug) {
		pid = fork();
		if (pid == -1)
			err(1, "fork");
		if (pid > 0)
			exit(0);
		setsid();
		pid = fork();
		if (pid == -1)
			err(1, "fork");
		if (pid > 0)
			exit(0);
		pid = getpid();
		umask(022);
		close(0);
		close(1);
		close(2);
		if (open("/dev/null", O_RDWR) == -1) {
			tell(LOG_INFO, "open: %s", strerror(errno));
			exit(1);
		}
		dup(0);
		dup(0);
	} else {
		debug("debug mode, not daemonizing");
	}

	/*
	 * Startup test.
	 */
	if (cfg.startup_test)
		ups_startup_test();

	/*
	 * Setup interval timer.
	 */
	t.it_value.tv_sec = APCD_QUERY_INT;
	t.it_value.tv_usec = 0;
	t.it_interval.tv_sec = APCD_QUERY_INT;
	t.it_interval.tv_usec = 0;
	r = setitimer(ITIMER_REAL, &t, NULL);
	if (r == -1) {
		tell(LOG_INFO, "setitimer: %s", strerror(errno));
		exit(1);
	}

	tell(LOG_INFO, "started");

	/*
	 * Create PID file.
	 */
	if (cfg.pidfile[0] != '\0') {
		f = fopen(cfg.pidfile, "w");
		if (f != NULL) {
			fclose(f);
			flog(cfg.pidfile, "%d\n", pid);
		} else {
			tell(LOG_INFO, "%s: %s", cfg.pidfile, strerror(errno));
		}
	}

	/*
	 * sleep loop (no listener active)
	 */
	quit = 0;
	while (!quit && cfg.listener == -1) {
		(void)sleep(APCD_POLL_TOUT / 1000);
	}
	if (cfg.listener == -1)
		goto quit;

	/*
	 * poll loop
	 */
	poll_init(fdpoll, sdlistener);
	quit = numclients = 0;
	sdclient = -1;
	while (!quit) {
		nfds = poll(fdpoll, APCD_POLL_FDS, APCD_POLL_TOUT);
		if (nfds == -1) {
			if (errno == EINTR)
				continue;
			tell(LOG_INFO, "poll: %s", strerror(errno));
			quit = 1;
		}
		if (nfds == 0) {
			tell(LOG_INFO, "poll timeout");
			continue;
		}

		if (fdpoll[0].revents & POLLIN) {
			/* new client connection arrived */
			sdclient = accept(sdlistener, (struct sockaddr *)&ca,
			    &sdsize);
			if (sdclient == -1) {
				err(1, "accept");
				quit = 1;
			}
			if (numclients == (APCD_POLL_FDS - 1)) {
				tell(LOG_INFO, "maximal clients reached");
				close(sdclient);
				continue;
			}
			poll_add(fdpoll, sdclient);
			numclients++;
			query.proto_numclients = numclients;
			debug("new client connection");
		}

		for (i = 1; i < APCD_POLL_FDS; i++) {
			if (fdpoll[i].revents & POLLIN) {
				sdclient = fdpoll[i].fd;
				n = client_request(sdclient);
				if (n < 1) {
					close(sdclient);
					poll_remove(fdpoll, sdclient);
					numclients--;
					query.proto_numclients = numclients;
					debug("client closed connection");
				}
			}
		}
	}
quit:
	tell(LOG_INFO, "stopped");

	return (0);
}

void
poll_init(struct pollfd *fdpoll, int sdlistener)
{
	int i;

	for (i = 0; i < APCD_POLL_FDS; i++) {
		if (i == 0)
			fdpoll[i].fd = sdlistener;
		else
			fdpoll[i].fd = -1;
		fdpoll[i].events = POLLIN;
	}
}

void
poll_add(struct pollfd *fdpoll, int sdclient)
{
	int i;

	for (i = 0; i < APCD_POLL_FDS; i++) {
		if (fdpoll[i].fd == -1) {
			fdpoll[i].fd = sdclient;
			break;
		}
	}
}

void
poll_remove(struct pollfd *fdpoll, int sdclient)
{
	int i;

	for (i = 0; i < APCD_POLL_FDS; i++) {
		if (fdpoll[i].fd == sdclient) {
			fdpoll[i].fd = -1;
			break;
		}
	}
}

int
ups_write(int ch)
{
	write(fdserial, &ch, 1);

	return (0);
}

int
ups_read(char *buf)
{
	int done, i;
	char c;

	done = i = 0;
	while (!done) {
		if (i == (APCD_MAXRESP - 1)) {
			debug("maximum response width exceeded!");
			return (-1);
		}
		read(fdserial, &c, 1);

		switch (c) {
		case '\n':
			done = 1;
			break;
		default:
			buf[i++] = c;
			break;
		}
	}
	buf[i] = '\0';

	return (0);
}

int
ups_query(struct ups_query *q)
{
	char buf[APCD_MAXRESP];

	/* make a copy of the last query values */
	querylast = query;

	ups_write('Y');
	ups_read(buf);

	ups_write('f');
	ups_read(buf);
	q->ups_batcharge = atof(buf);

	ups_write('N');
	ups_read(buf);
	q->ups_linemin = atof(buf);

	ups_write('M');
	ups_read(buf);
	q->ups_linemax = atof(buf);

	ups_write('P');
	ups_read(buf);
	q->ups_load = atof(buf);

	ups_write('F');
	ups_read(buf);
	q->ups_linefreq = atof(buf);

	ups_write('L');
	ups_read(buf);
	q->ups_linevolt = atof(buf);

	ups_write('O');
	ups_read(buf);
	q->ups_outvolt = atof(buf);

	ups_write('C');
	ups_read(buf);
	q->ups_temp = atof(buf);

	ups_write('B');
	ups_read(buf);
	q->ups_batvolt = atof(buf);

	ups_write('Q');
	ups_read(buf);
	q->ups_status = atoi(buf);

	return (0);
}

void
ups_startup_test(void)
{
	char buf[APCD_MAXRESP];

	ups_write('Y');
	ups_read(buf);

	ups_write('A');
	ups_read(buf);

	tell(LOG_INFO, "performed startup test");
	sleep(1);
}

void
ups_check_condition(void)
{
	switch (query.ups_status) {
	case UPS_STATUS_ONLINE:
		if (querylast.ups_status == UPS_STATUS_BATTERY)
			alert_online();
		break;
	case UPS_STATUS_BATTERY:
		if (querylast.ups_status == UPS_STATUS_ONLINE)
			alert_battery();
		break;
	case UPS_STATUS_SHUTDOWN:
		if (querylast.ups_status != UPS_STATUS_SHUTDOWN)
			alert_shutdown();
		break;
	default:
		break;
	}

	if (query.ups_batcharge == cfg.battery_low) {
		if (querylast.ups_batcharge > cfg.battery_low)
			alert_battery_low();
	}

	if (query.ups_batcharge == cfg.battery_crit) {
		if (querylast.ups_batcharge > cfg.battery_crit)
			alert_battery_crit();
	}
}

void
alert_online(void)
{
	tell(LOG_INFO, "UPS switched back online");

	if (cfg.trap_online[0] != '\0')
		trap_run(cfg.trap_online);
}

void
alert_battery(void)
{
	tell(LOG_INFO, "UPS switched to battery");

	if (cfg.trap_battery[0] != '\0')
		trap_run(cfg.trap_battery);
}

void
alert_battery_low(void)
{
	tell(LOG_INFO, "battery is low (%6.2f %%)", cfg.battery_low);

	if (cfg.trap_battery_low[0] != '\0')
		trap_run(cfg.trap_battery_low);
}

void
alert_battery_crit(void)
{
	tell(LOG_INFO, "battery is critical (%6.2f %%)", cfg.battery_crit);

	if (cfg.trap_battery_crit[0] != '\0')
		trap_run(cfg.trap_battery_crit);
}

void
alert_shutdown(void)
{
	tell(LOG_INFO, "UPS is shutting down");

	if (cfg.trap_shutdown[0] != '\0')
		trap_run(cfg.trap_shutdown);
}

void
trap_run(char *cmd)
{
	pid_t pid;

	/* execute command in child process */
	pid = fork();
	if (pid == -1) {
		tell(LOG_INFO, "fork: %s", strerror(errno));
		return;
	}
	if (pid == 0) {
		close(0);
		close(1);
		close(2);
		system(cmd);
		exit(0);
	}

	debug("%s: %s", __func__, cmd);
}

int
client_request(int sdclient)
{
	int n;
	char reqbuf[APCD_REQBUFSIZE];

	n = read(sdclient, reqbuf, sizeof(reqbuf));
	if (n < 1)
		return (n);

	if (!strncasecmp(reqbuf, "get\r\n", 5)) {
		write(sdclient, (char *)&query, sizeof(query));
	}

	return (n);
}

int
load_config(int daemon)
{
	int r;
	char tmp[128];
	struct stat s;

	/*
	 * Parse configuration file.
	 */
	r = stat(cfgfile, &s);
	if (r == -1) {
		if (daemon)
			tell(LOG_INFO, "%s: %s", cfgfile, strerror(errno));
		else
			warn("%s", cfgfile);
		return (r);
	}

	r = fparse(cfg.device, "device", cfgfile, sizeof(cfg.device));
	if (r == -1) {
		if (daemon)
			tell(LOG_INFO, "%s: 'device' not found", cfgfile);
		else
			fprintf(stderr, "%s: 'device' not found.\n", cfgfile);
		return (r);
	}

	r = fparse(cfg.pidfile, "pidfile", cfgfile, sizeof(cfg.pidfile));
	if (r == -1)
		cfg.pidfile[0] = '\0';

	r = fparse(tmp, "startup_test", cfgfile, sizeof(cfg.startup_test));
	if (r) {
		if (!strcasecmp(tmp, "yes"))
			cfg.startup_test = 1;
		else
			cfg.startup_test = 0;
	}

	r = fparse(tmp, "listener", cfgfile, sizeof(tmp));
	if (r == -1)
		cfg.listener = -1;
	else
		cfg.listener = atoi(tmp);

	r = fparse(tmp, "battery_low", cfgfile, sizeof(tmp));
	if (r == -1)
		cfg.battery_low = -1;
	else
		cfg.battery_low = atof(tmp);

	r = fparse(tmp, "battery_critical", cfgfile, sizeof(tmp));
	if (r == -1)
		cfg.battery_crit = -1;
	else
		cfg.battery_crit = atof(tmp);

	r = fparse(cfg.trap_battery, "trap-on_battery", cfgfile,
	    sizeof(cfg.trap_battery));
	if (r == -1)
		cfg.trap_battery[0] = '\0';

	r = fparse(cfg.trap_online, "trap-back_online", cfgfile,
	    sizeof(cfg.trap_online));
	if (r == -1)
		cfg.trap_online[0] = '\0';

	r = fparse(cfg.trap_battery_low, "trap-battery_low", cfgfile,
	    sizeof(cfg.trap_battery_low));
	if (r == -1)
		cfg.trap_battery_low[0] = '\0';

	r = fparse(cfg.trap_battery_crit, "trap-battery_critical", cfgfile,
	    sizeof(cfg.trap_battery_crit));
	if (r == -1)
		cfg.trap_battery_crit[0] = '\0';

	r = fparse(cfg.trap_shutdown, "trap-shutdown", cfgfile,
	    sizeof(cfg.trap_shutdown));
	if (r == -1)
		cfg.trap_shutdown[0] = '\0';

	return (0);
}

void
tell(int prio, char *msg, ...)
{
	char buf[1024];
	va_list ap;

	if (optdebug) {
		va_start(ap, msg);
		vprintf(msg, ap);
		va_end(ap);
		printf("\n");
	} else {
		va_start(ap, msg);
		vsnprintf(buf, sizeof(buf), msg, ap);
		syslog(prio, "%s", buf);
		va_end(ap);
	}
}

void
debug(char *msg, ...)
{
	va_list ap;

	if (optdebug) {
		va_start(ap, msg);
		vprintf(msg, ap);
		va_end(ap);
		printf("\n");
	}
}
