/*
 * Copyright (c) 2009 Marcus Glocker <marcus@nazgul.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <netinet/in.h>

#include <err.h>
#include <errno.h>
#include <curses.h>
#include <fcntl.h>
#include <poll.h>
#include <netdb.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <termios.h>
#include <time.h>
#include <unistd.h>

#include "../apc.h"

/*
 * Prototypes
 */
void	signal_handler(int);
void	usage(void);
void	display_console(void);
void	display_curses(void);
void	display_curses_refresh(void);
int	server_query(void);

/*
 * Global variables
 */
extern char		*__progname;
int			 sdserver;
int			 optconsole;
char			 hostname[128];
char			 buf[256];
struct ups_query	*query = (struct ups_query *)buf;

void
signal_handler(int sig)
{
	switch (sig) {
	case SIGALRM:
		bzero(buf, sizeof(buf));
		server_query();
		break;
	default:
		break;
	}
}

void
usage(void)
{
	extern char *__progname;

	fprintf(stderr, "usage: %s ", __progname);
	fprintf(stderr, "[-c] hostname\n");
	exit(1);
}

int
main(int argc, char *argv[])
{
	int ch, r, port;
	struct hostent *host;
	struct sockaddr_in sa;

	/*
	 * Parse command line options
	 */
	while ((ch = getopt(argc, argv, "chv")) != -1) {
		switch (ch) {
		case 'c':
			optconsole = 1;
			break;
		case 'h':
			usage();
			/* NOTREACHED */
		case 'v':
			printf("%s 0.2\n", __progname);
			return (0);
			break;
		default:
			usage();
			/* NOTREACHED */
		}
	}
	argc -= optind;
	argv += optind;
	if (argc < 1)
		usage();
	strlcpy(hostname, argv[0], sizeof(hostname));
	if (argv[1] != NULL) {
		strlcpy(buf, argv[1], sizeof(buf));
		port = atoi(buf);
	} else
		port = APCD_LISTEN_PRT;

	signal(SIGALRM, signal_handler);

	/*
	 * Connect the UPS daemon
	 */
	host = gethostbyname(hostname);
	if (host == NULL)
		err(1, "gethostbyname");

	sdserver = socket(AF_INET, SOCK_STREAM, 0);
	if (sdserver == -1)
		err(1, "socket");

	sa.sin_family = AF_INET;
	sa.sin_port = htons(port);
	sa.sin_addr = *((struct in_addr *)host->h_addr);
	memset(&(sa.sin_zero), 0, sizeof(sa.sin_zero));
	r = connect(sdserver, (struct sockaddr *)&sa,sizeof(struct sockaddr));
	if (r == -1)
		err(1, "connect");

	if (optconsole)
		display_console();
	else
		display_curses();

	return (0);
}

void
display_console(void)
{
	bzero(buf, sizeof(buf));
	if (server_query() == -1)
		err(1, "server_query");

	printf("Temperature     : %6.2f degC\n",
	    query->ups_temp);
	printf("Load            : %6.2f %%\n",
	    query->ups_load);
	printf("Battery charged : %6.2f %%\n",
	    query->ups_batcharge);
	printf("Battery voltage : %6.2f volt\n",
	    query->ups_batvolt);
	printf("Voltage         : %6.2f volt\n",
	    query->ups_linevolt);
	printf("Output voltage  : %6.2f volt\n",
	    query->ups_outvolt);
        printf("Line minimum    : %6.2f volt\n",
	    query->ups_linemin);
        printf("Line maximum    : %6.2f volt\n",
	    query->ups_linemax);
	printf("Line frequence  : %6.2f HZ\n",
	    query->ups_linefreq);
	printf("Status          : %02d ", query->ups_status);
	switch (query->ups_status) {
	case UPS_STATUS_SHUTDOWN:
		printf("(SHUTDOWN STATE)");
		break;
	case UPS_STATUS_ONLINE:
		printf("(ON-LINE)");
		break;
	case UPS_STATUS_BATTERY:
		printf("(ON-BATTERY)");
		break;
	case UPS_STATUS_BOTH:
		printf("(ON-LINE / ON-BATTERY)");
		break;
	default:
		printf("(UNKNOWN)");
		break;
	}
	printf("\n");
}

void
display_curses(void)
{
	int nfds, quit, error, r;
	char ch;
	struct pollfd fdpoll[1];
	struct itimerval t;

	nfds = quit = error = r = 0;

	initscr();
	clear();
	refresh();

	/*
	 * Setup interval timer.
	 */
	t.it_value.tv_sec = APCD_QUERY_INT;
	t.it_value.tv_usec = 0;
	t.it_interval.tv_sec = APCD_QUERY_INT;
	t.it_interval.tv_usec = 0;
	r = setitimer(ITIMER_REAL, &t, NULL);
	if (r == -1) {
		err(1, "setitimer");
		return;
	}

	fdpoll[0].fd = STDIN_FILENO;
	fdpoll[0].events = POLLIN;
	while (!quit) {
		display_curses_refresh();

		nfds = poll(fdpoll, 1, 1000);
		if (nfds == -1) {
			if (errno == EINTR)
				continue;
			break;
		}
		if (nfds == 0)
			continue;

		if (fdpoll[0].revents & POLLIN) {
			/* handle keyboard input */
			read(STDIN_FILENO, &ch, 1);

			switch (ch) {
			case 'q':
				quit = 1;
				break;
			default:
				break;
			}
		}
	}

	endwin();
}

void
display_curses_refresh(void)
{
	time_t tnow;
	struct tm *t;

	/* header 1 */
	time(&tnow);
	t = localtime(&tnow);
	mvprintw(0, 0, "On Host: %s",
	    hostname);
	mvprintw(0, 55, "Clients: %d",
	    query->proto_numclients);
	mvprintw(0, 72, "%02d:%02d:%02d",
	    t->tm_hour, t->tm_min, t->tm_sec);

	/* UPS infos */
	mvprintw(2, 9, "Temperature     : %6.2f degC",
	    query->ups_temp);
	mvprintw(3, 9, "Load            : %6.2f %%",
	    query->ups_load);
	mvprintw(4, 9, "Battery charged : %6.2f %%",
	    query->ups_batcharge);
	mvprintw(5, 9, "Battery voltage : %6.2f volt",
	    query->ups_batvolt);
	mvprintw(6, 9, "Voltage         : %6.2f volt",
	    query->ups_linevolt);
	mvprintw(7, 9, "Output voltage  : %6.2f volt",
	    query->ups_outvolt);
	mvprintw(8, 9, "Line minimum    : %6.2f volt",
	    query->ups_linemin);
	mvprintw(9, 9, "Line maximum    : %6.2f volt",
	    query->ups_linemax);
	mvprintw(10, 9, "Line frequence  : %6.2f HZ",
	    query->ups_linefreq);
	mvprintw(11, 9, "Status          : %02d ",
	    query->ups_status);
	switch (query->ups_status) {
	case UPS_STATUS_SHUTDOWN:
		mvprintw(11, 30, "(SHUTDOWN STATE)      ");
		break;
	case UPS_STATUS_ONLINE:
		mvprintw(11, 30, "(ON-LINE)             ");
		break;
	case UPS_STATUS_BATTERY:
		mvprintw(11, 30, "(ON-BATTERY)          ");
		break;
	case UPS_STATUS_BOTH:
		mvprintw(11, 30, "(ON-LINE / ON-BATTERY)");
		break;
	default:
		mvprintw(11, 30, "(UNKNOWN)             ");
		break;
	}
	mvprintw(1, 0, "");
	refresh();
}

int
server_query(void)
{
	int n;

	n = write(sdserver, "get\r\n", 5);
	if (n < 5)
		return (-1);

	n = read(sdserver, buf, sizeof(struct ups_query));
	if (n < 1)
		return (-1);

	return (n);
}
