/*	$Id: spaceinvaders.c,v 1.3 2017/05/16 17:27:04 hacki Exp $ */

/*
 * Copyright (c) 2017 Marcus Glocker <marcus@nazgul.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
 * Another version of the famous Space Invaders arcade video game created
 * by Tomohiro Nishikado and published in 1978.  I tried to retain the
 * original game rules as far as possible.
 */

#include <curses.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <poll.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include "spaceinvaders.h"

/*
 * Prototypes.
 */
void	usage(int);
int	intro(void);
void	timer_setup(void);
void	timer_clear(void);
void	keyboard_handler(void);
void	hiscore_load(void);
void	hiscore_save(void);
void	ctrl_init(int);
void	board_init(int);
void	board_draw(void);
void	board_write(int, int, int, const char *, ...);
void	board_update(int);
void	board_update_player_move(int);
void	board_update_player_shot(void);
void	board_update_player_laser(void);
int	board_update_invaders_move(void);
void	board_update_invaders_shot(void);
int	board_update_invaders_missle(void);
void	board_update_mystery_move(void);
void	board_update_mystery_trigger(void);
void	board_backing_store(int);
int	lookup_sprite(char);
void	player_score(int);
void	player_destroy(void);
void	player_redraw_lives(void);
void	game_reset(int);
void	game_pause(void);
void	game_over(void);
char	getkey(void);
void	flush_stdin(void);
void	debug_setup(void);
void	debug_write(char *, ...);

int
main(int argc, char *argv[])
{
	int ch;

	while ((ch = getopt(argc, argv, "dhv")) != -1) {
		switch (ch) {
		case 'h':
			usage(0);
			/* NOTREACHED */
		case 'd':
			/* Undocumented. */
			debug_setup();
			break;
		case 'v':
			usage(1);
			/* NOTREACHED */
		default:
			usage(0);
			/* NOTREACHED */
		}
	}

	/* Init curses. */
	initscr();

	/* Handle signals. */
	signal(SIGALRM, board_update);
	signal(SIGINT, SIG_IGN);

	/* Make STDIN non-blocking. */
	fcntl(STDIN_FILENO, F_SETFL, fcntl(0, F_GETFL) | O_NONBLOCK);

	/* Load hi-score. */
	hiscore_load();

	/* Display intro. */
	if (intro()) {
		/* Start the game. */
		DM("Game has started.");
		game_reset(0);
		keyboard_handler();
		DM("Game has stopped.");
	}

	/* End curses. */
	endwin();

	return 0;
}

void
usage(int mode)
{
	if (mode) {
		fprintf(stderr, "spaceinvaders version %s\n", VERSION);
	} else {
		fprintf(stderr, "usage: %s ", __progname);
		fprintf(stderr, "[-hv]\n");
	}
	exit(1);
}

/*
 * RETURN codes:
 *	0 = Quit.
 *	1 = Start the game.
 */
int
intro(void)
{
	char ch = 0;

	board_init(1);

	board_write(3,  39, 0, "PLAY");
	board_write(5,  34, 0, "SPACE INVADERS");
	board_write(7,  30, 0, "*SCORE ADVANCE TABLE*");
	board_write(8,  34, 1, "T = ? Mystery");
	board_write(9,  34, 1, "Y = 30 Points");
	board_write(10, 34, 1, "O = 20 Points");
	board_write(11, 34, 1, "H = 10 Points");
	board_write(13, 31, 0, "*GAME CONTROL KEYS*");
	board_write(14, 22, 0, "CURSOR-LEFT / H = Move Laser Base Left");
	board_write(15, 21, 0, "CURSOR-RIGHT / L = Move Laser Base Right");
	board_write(16, 26, 0, "CURSOR-UP / SPACE = Laser Shot");
	board_write(17, 34, 0, "P = Pause Game");
	board_write(18, 34, 0, "Q = Quit Game");
	board_write(21, 22, 0, "PRESS S TO START THE GAME OR Q TO QUIT");

	board_draw();

	/* Wait for user to start game. */
	while (1) {
		ch = getkey();

		switch (ch) {
		case 's':
			return 1;
			/* NOTREACHED */
		case 'q':
			return 0;
			/* NOTREACHED */
		}
	}
}

void
timer_setup(void)
{
	struct itimerval t;

	t.it_interval.tv_sec = 0;
	t.it_interval.tv_usec = BOARD_UPDATE_USEC;
	t.it_value.tv_sec = 0;
	t.it_value.tv_usec = 1; /* Start timer almost immediately. */
	if (setitimer(ITIMER_REAL, &t, NULL) == -1)
		err(1, "setitimer");
}

void
timer_clear(void)
{
	struct itimerval t;

	t.it_interval.tv_sec = 0;
	t.it_interval.tv_usec = 0;
	t.it_value.tv_sec = 0;
	t.it_value.tv_usec = 0;
	if (setitimer(ITIMER_REAL, &t, NULL) == -1)
		err(1, "setitimer");
}

void
keyboard_handler(void)
{
	int quit = 0;
	char ch = 0;

	while (!quit) {
		ch = getkey();
		lock_signal++;

		if (ctrl.game_pause) {
			switch (ch) {
			case 'p':
				/* Handle in main switch. */
				break;
			default:
				/* No other keys accepted. */
				lock_signal--;
				continue;
			}
		}

		if (ctrl.game_over) {
			switch (ch) {
			case 's':
				/* Start new game. */
				game_reset(0);
				lock_signal--;
				continue;
			case 'q':
				/* Handle in main switch. */
				break;
			default:
				/* No other keys accepted. */
				lock_signal--;
				continue;
			}
		}

		switch (ch) {
		case 67:	/* Cursor right. */
		case 'l':
			/* Move playership to the right. */
			board_update_player_move(MOVE_PLAYER_LB_RIGHT);
			break;
		case 68:	/* Cursor left. */
		case 'h':
			/* Move playership to the left. */
			board_update_player_move(MOVE_PLAYER_LB_LEFT);
			break;
		case 65:	/* Cursor up. */
		case ' ':
			/* Playership shots. */
			board_update_player_shot();
			break;
		case 'p':
			/* Pause or resume the game. */
			game_pause();
			break;
		case 'q':
			/* Quit the game. */
			hiscore_save();
			quit = 1;
			break;
		default:
			/* No other keys accepted. */
			break;
		}
		lock_signal--;
	}
}

void
hiscore_load(void)
{
	int hiscore_fd;
	char hiscore[5];
	char filepath[1024];
	char *home;
	const char *errstr = NULL;

	home = getenv("HOME");
	snprintf(filepath, sizeof(filepath), "%s/%s", home, FILENAME_HISCORE);

	hiscore_fd = open(filepath, O_RDONLY);
	if (hiscore_fd == -1) {
		/* Can't open hiscore file. */
		ctrl.player_hiscore = 0;
		return;
	}

	memset(hiscore, 0, sizeof(hiscore));
	read(hiscore_fd, hiscore, sizeof(hiscore)-1);
#ifdef __OpenBSD__
	ctrl.player_hiscore = strtonum(hiscore, 0, SCORE_MAX, &errstr);
#else
	ctrl.player_hiscore = atoi(hiscore);
#endif
	if (errstr)
		err(1, "%s", errstr);
	close(hiscore_fd);
}

void
hiscore_save(void)
{
	int hiscore_fd;
	char hiscore[5];
	char filepath[1024];
	char *home;

	home = getenv("HOME");
	snprintf(filepath, sizeof(filepath), "%s/%s", home, FILENAME_HISCORE);

	hiscore_fd = open(filepath, O_CREAT|O_TRUNC|O_WRONLY, 0644);
	if (hiscore_fd == -1) {
		/* Can't open hiscore file. */
		return;
	}

	snprintf(hiscore, sizeof(hiscore), "%04d", ctrl.player_hiscore);
	write(hiscore_fd, hiscore, 4);
	close(hiscore_fd);
}

void
ctrl_init(int mode)
{
	if (mode == 0) {
		ctrl.player_score = 0;
		ctrl.player_lives = PLAYER_LIVES;
		ctrl.score_extra_live = SCORE_EXTRA_LIVE;
		ctrl.player_hiscore_new = 0;
	}
	ctrl.player_pos_x = 14;
	ctrl.player_shot_inprogress = 0;
	ctrl.player_laser_cycle = 0;
	ctrl.invaders_num = INVADERS_NUM;
	ctrl.invaders_shot_inprogress = 0;
	ctrl.invaders_dir_now = MOVE_INVADERS_RIGHT;
	ctrl.invaders_dir_last = MOVE_INVADERS_RIGHT;
	ctrl.invaders_move_cycle = 0;
	ctrl.invaders_move_usec = INVADERS_MOVE_USEC;
	ctrl.invaders_shot_cycle = 0;
	ctrl.invaders_missle_cycle = 0;
	ctrl.mystery_trigger_cycle = 0;
	ctrl.mystery_move_cycle = 0;
	ctrl.mystery_inprogress = 0;
	ctrl.text_cleanup_cycle = 0;
	ctrl.game_over = 0;
	ctrl.game_pause = 0;
}

void
board_init(int intro)
{
	int y, x, i;

	clear();

	/* Initialize board with spaces. */
	for (y = 0; y < BOARD_ROWS; y++)
		memset(board[y], sprites[SPACE].graph, BOARD_COLS);

	/* Scores. */
	board_write(0, 0, 0, "%s", text_score);
	board_write(0, 16, 0, "%s", text_hiscore);
	board_write(1, 0, 0, "%04d", ctrl.player_score);
	board_write(1, 16, 0, "%04d", ctrl.player_hiscore);

	if (intro)
		return;

	/* Invader 1. */
	for (y = 0, x = 18; y < INVADERS_PER_LINE; y++, x += 4)
		board[4][x] = sprites[INVADER_1].graph;

	/* Invader 2. */
	for (y = 0, x = 18; y < INVADERS_PER_LINE; y++, x += 4)
		board[5][x] = sprites[INVADER_2].graph;
	for (y = 0, x = 18; y < INVADERS_PER_LINE; y++, x += 4)
		board[6][x] = sprites[INVADER_2].graph;

	/* Invader 3. */
	for (y = 0, x = 18; y < INVADERS_PER_LINE; y++, x += 4)
		board[7][x] = sprites[INVADER_3].graph;
	for (y = 0, x = 18; y < INVADERS_PER_LINE; y++, x += 4)
		board[8][x] = sprites[INVADER_3].graph;

	/* Player defense. */
	for (y = 0, x = 12; y < 4; y++, x += 16)
		memset(board[17]+x, sprites[SHIELD_3].graph, 6);
	for (y = 0, x = 11; y < 4; y++, x += 16)
		memset(board[18]+x, sprites[SHIELD_3].graph, 8);
	for (y = 0, x = 11, i = 0; y < 8; y++, x += i) {
		memset(board[19]+x, sprites[SHIELD_3].graph, 2);
		if (i == 0)
			i = 6;
		else if (i == 6)
			i = 10;
		else if (i == 10)
			i = 6;
	}

	/* Players Laser Base. */
	board[21][ctrl.player_pos_x] = sprites[PLAYER_LASER_BASE].graph;

	/* Floor. */
	memset(board[22], sprites[FLOOR].graph, BOARD_COLS);

	/* Player lives. */
	board_write(23, 0, 0, "%d", ctrl.player_lives);
	for (y = 0, x = 8; y < (ctrl.player_lives - 1); y++, x += 2)
		board[23][x] = sprites[PLAYER_LASER_BASE].graph;

	board_draw();
}

void
board_draw(void)
{
	int y, x;

	for (y = 0; y < BOARD_ROWS; y++) {
		for (x = 0; x < BOARD_COLS; x++)
			mvaddch(y, x, board[y][x]);
	}
	move(23, 79);
	refresh();
}

void
board_write(int y, int x, int cleanup, const char *msg, ...)
{
	int i;
	char buf[1024];
	va_list ap;

	va_start(ap, msg);
	vsnprintf(buf, sizeof(buf), msg, ap);
	va_end(ap);

	for (i = 0; i < strlen(buf); i++, x++)
		board[y][x] = buf[i];

	if (cleanup)
		ctrl.text_cleanup_cycle = 0;
}

void
board_update(int sig)
{
	int r1 = 0;
	int r2 = 0;

	if (sig != SIGALRM) {
		/* We just care about the timer signal. */
		return;
	}
	if (lock_signal) {
		/* Signal handler got locked, try again later. */
		DM("Signal handler is locked.");
		return;
	}

	/* Update cycle timers. */
	ctrl.invaders_move_cycle += BOARD_UPDATE_USEC;
	ctrl.invaders_missle_cycle += BOARD_UPDATE_USEC;
	ctrl.invaders_shot_cycle += BOARD_UPDATE_USEC;
	ctrl.player_laser_cycle += BOARD_UPDATE_USEC;
	ctrl.mystery_trigger_cycle += BOARD_UPDATE_USEC;
	ctrl.mystery_move_cycle += BOARD_UPDATE_USEC;
	ctrl.text_cleanup_cycle += BOARD_UPDATE_USEC;

	if (ctrl.invaders_move_cycle >= ctrl.invaders_move_usec) {
		/* Update invaders movement. */
		ctrl.invaders_move_cycle = 0;
		r1 = board_update_invaders_move();
		if (r1 == 1) {
			/* Change moving direction. */
			r1 = board_update_invaders_move();
		}
		if (r1 == 2) {
			/* Invader touches floor. */
			game_over();
			return;
		}
	}
	if (ctrl.invaders_missle_cycle == INVADERS_MISSLE_USEC) {
		ctrl.invaders_missle_cycle = 0;
		/* Update invaders missle. */
		r2 = board_update_invaders_missle();
	}
	if (ctrl.invaders_shot_cycle == INVADERS_SHOT_USEC) {
		/* Trigger invader shot. */
		ctrl.invaders_shot_cycle = 0;
		board_update_invaders_shot();
	}
	if (ctrl.player_laser_cycle == PLAYER_LASER_USEC) {
		/* Update players laser. */
		ctrl.player_laser_cycle = 0;
		board_update_player_laser();
	}
	if (ctrl.mystery_trigger_cycle == MYSTERY_TRIGGER_USEC) {
		/* Trigger mystery ship to appear. */
		ctrl.mystery_trigger_cycle = 0;
		board_update_mystery_trigger();
	}
	if (ctrl.mystery_move_cycle == MYSTERY_MOVE_USEC) {
		/* Update mystery ship movement. */
		ctrl.mystery_move_cycle = 0;
		board_update_mystery_move();
	}
	if (ctrl.text_cleanup_cycle == TEXT_CLEANUP_USEC) {
		/* Cleanup text messages. */
		ctrl.text_cleanup_cycle = 0;
		memset(board[0]+35, sprites[SPACE].graph, 39);
	}

	if (ctrl.invaders_move_cycle == 0 ||
	    ctrl.invaders_shot_cycle == 0 ||
	    ctrl.invaders_missle_cycle == 0 ||
	    ctrl.player_laser_cycle == 0 ||
	    ctrl.mystery_trigger_cycle == 0 ||
	    ctrl.mystery_move_cycle == 0 ||
	    ctrl.text_cleanup_cycle == 0) {
		/* Refresh screen if update is pending. */
		board_draw();
	}

	/* Make a quick break when the player gets destroyed. */
	if (r2)
		player_destroy();
}

void
board_update_player_move(int action)
{
	int next_sprite;
	int old_player_pos_x = ctrl.player_pos_x;

	if (action == MOVE_PLAYER_LB_RIGHT) {
		if (ctrl.player_pos_x == (BOARD_COLS-1)) {
			/* Player touches right border. */
			return;
		}
		ctrl.player_pos_x++;
	}
	if (action == MOVE_PLAYER_LB_LEFT) {
		if (ctrl.player_pos_x == 0) {
			/* Player touches the left border. */
			return;
		}
		ctrl.player_pos_x--;
	}

	next_sprite = lookup_sprite(board[21][ctrl.player_pos_x]);
	if (next_sprite == INVADER_1 ||
	    next_sprite == INVADER_2 ||
	    next_sprite == INVADER_3) {
		/* Player touches invader. */
		DM("Player touches invader.");
		game_over();
		return;
	}

	/* Clear old position. */
	board[21][old_player_pos_x] = sprites[SPACE].graph;

	/* Set new position. */
	board[21][ctrl.player_pos_x] = sprites[PLAYER_LASER_BASE].graph;

	board_draw();
}

void
board_update_player_shot(void)
{
	if (ctrl.player_shot_inprogress) {
		/* Another shot still in progress. */
		return;
	}
	board[20][ctrl.player_pos_x] = sprites[PLAYER_LASER].graph;
	ctrl.player_shot_inprogress++;

	board_draw();
}

void
board_update_player_laser(void)
{
	int y, x, sprite, sprite_next;
	int defense_damage;

	for (y = 3; y < (BOARD_ROWS-2); y++) {
		for (x = 0; x < BOARD_COLS; x++) {
			sprite = lookup_sprite(board[y][x]);
			if (sprite != PLAYER_LASER)
				continue;

			if ((y-1) == 2) {
				/* Player hits nothing. */
				board[y][x] = sprites[SPACE].graph;
				ctrl.player_shot_inprogress--;
				continue;
			}

			sprite_next = lookup_sprite(board[y-1][x]);

			if (sprite_next == SPACE) {
				/* The way is free. */
				board[y][x] = sprites[SPACE].graph;
				board[y-1][x] = sprites[PLAYER_LASER].graph;
			}

			if (sprite_next == INVADER_1_MISSLE ||
			    sprite_next == INVADER_2_MISSLE ||
			    sprite_next == INVADER_3_MISSLE) {
				/* Player shot passes invader shot. */
				DM("Player shot passes invader shot.");
				board[y][x] = sprites[sprite_next].graph;
				board[y-1][x] = sprites[sprite].graph;
			}

			if (sprite_next == INVADER_1 ||
			    sprite_next == INVADER_2 ||
			    sprite_next == INVADER_3) {
				/* Player destroys invader. */
				DM("Player destroys invader.");
				board[y][x] = sprites[SPACE].graph;
				board[y-1][x] = sprites[SPACE].graph;
				ctrl.player_shot_inprogress--;
				player_score(sprite_next);
			}

			if (sprite_next == MYSTERY_SHIP) {
				/* Player destroys mystery ship. */
				DM("Player destroys mystery ship.");
				board[y][x] = sprites[SPACE].graph;
				board[y-1][x] = sprites[SPACE].graph;
				ctrl.player_shot_inprogress--;
				ctrl.mystery_inprogress--;
				player_score(sprite_next);
			}

			if (sprite_next == SHIELD_3 ||
			    sprite_next == SHIELD_2 ||
			    sprite_next == SHIELD_1) {
				/* Player damages defense. */
				defense_damage =
				    sprites[sprite_next].protection -
				    sprites[sprite].damage;
				switch (defense_damage) {
				case 2:
					/* Player defense damaged. */
					board[y-1][x] =
					    sprites[SHIELD_2].graph;
					break;
				case 1:
					/* Player defense weak. */
					board[y-1][x] =
					    sprites[SHIELD_1].graph;
					break;
				default:
					/* Player defense cracked. */
					board[y-1][x] =
					    sprites[SPACE].graph;
					break;
				}
				board[y][x] = sprites[SPACE].graph;
				ctrl.player_shot_inprogress--;
			}
		}
	}
}

/*
 * RETURN codes:
 *	0 = Regular move done.
 *	1 = Invaders touches border and need to change the direction.
 *	2 = Invaders touches floor which means the game is over.
 */
int
board_update_invaders_move(void)
{
	int y, x, sprite, sprite_next;

	board_backing_store(BBS_BACKUP);
	board_backing_store(BBS_REMOVE_INVADERS);

	for (y = 3; y < (BOARD_ROWS-2); y++) {
		for (x = 0; x < BOARD_COLS; x++) {
			sprite = lookup_sprite(board[y][x]);
			if (sprite != INVADER_1 &&
			    sprite != INVADER_2 &&
			    sprite != INVADER_3)
				continue;

			if (ctrl.invaders_dir_now == MOVE_INVADERS_RIGHT) {
				if ((x+1) == BOARD_COLS) {
					/* Invader touches the right border. */
					ctrl.invaders_dir_now =
					    MOVE_INVADERS_DOWN;
					return 1;
				}

				sprite_next = lookup_sprite(board[y][x+1]);
				if (sprite_next == PLAYER_LASER) {
					/* Playershot gets eliminated. */
					ctrl.player_shot_inprogress--;
				} else if (sprite_next == PLAYER_LASER_BASE) {
					/* Invader touches player. */
					return 2;
				}

				board_bs[y][x+1] = board[y][x];
			}

			if (ctrl.invaders_dir_now == MOVE_INVADERS_LEFT) {
				if ((x-1) < 0) {
					/* Invader touches the left border. */
					ctrl.invaders_dir_now =
					    MOVE_INVADERS_DOWN;
					return 1;
				}

				sprite_next = lookup_sprite(board[y][x-1]);
				if (sprite_next == PLAYER_LASER) {
					/* Playershot gets eliminated. */
					ctrl.player_shot_inprogress--;
				} else if (sprite_next == PLAYER_LASER_BASE) {
					/* Invader touches player. */
					return 2;
				}

				board_bs[y][x-1] = board[y][x];
			}

			if (ctrl.invaders_dir_now == MOVE_INVADERS_DOWN) {
				sprite_next = lookup_sprite(board_bs[y+1][x]);
				if (sprite_next == PLAYER_LASER) {
					/* Player destroys invader. */
					ctrl.player_shot_inprogress--;
					board[y][x] = sprites[SPACE].graph;
					player_score(sprite);
				} else if (sprite_next == PLAYER_LASER_BASE ||
					   sprite_next == FLOOR) {
					/* Invader touches floor. */
					return 2;
				}

				board_bs[y+1][x] = board[y][x];
			}
		}
	}

	if (ctrl.invaders_dir_now == MOVE_INVADERS_DOWN) {
		/* Change moving direction. */ 
		if (ctrl.invaders_dir_last == MOVE_INVADERS_RIGHT) {
			ctrl.invaders_dir_now = MOVE_INVADERS_LEFT;
			ctrl.invaders_dir_last = MOVE_INVADERS_LEFT;
		} else if (ctrl.invaders_dir_last == MOVE_INVADERS_LEFT) {
			ctrl.invaders_dir_now = MOVE_INVADERS_RIGHT;
			ctrl.invaders_dir_last = MOVE_INVADERS_RIGHT;
		}
	}

	board_backing_store(BBS_RESTORE);

	return 0;
}

void
board_update_invaders_shot(void)
{
	int y, x, i, skip, sprite, sprite_next;
	int invaders_num; /* Number of invaders who could shot. */
	int invader_shots; /* Invader we've choosen to shot. */
	struct position {
		int sprite;
		int y;
		int x;
	} pos[INVADERS_PER_LINE];

	if (ctrl.invaders_shot_inprogress == INVADERS_SHOT_MAX) {
		/* Invaders maximum parallel shots reached. */
		DM("Invaders maximum parallel shots reached.");
		return;
	}

	invaders_num = 0;
	memset(&pos, 0, sizeof(struct position));

	for (y = 3; y < (BOARD_ROWS-2); y++) {
		for (x = 0; x < BOARD_COLS; x++) {
			sprite = lookup_sprite(board[y][x]);
			if (sprite != INVADER_1 &&
			    sprite != INVADER_2 &&
			    sprite != INVADER_3)
				continue;

			sprite_next = lookup_sprite(board[y+1][x]);
			if (sprite_next == SPACE) {
				pos[invaders_num].sprite = sprite;
				pos[invaders_num].y = y+1;
				pos[invaders_num].x = x;
				invaders_num++;
			}
		}
	}
	if (invaders_num == 0) {
		/* No invader can shot currently. */
		return;
	}

	for (i = 0; i < INVADERS_SHOT_MAX; i++) {
		skip = arc4random_uniform(2);
		if (skip) {
			/* Invader don't always shot. */
			continue;
		}
		invader_shots = arc4random_uniform(invaders_num);
		y = pos[invader_shots].y;
		x = pos[invader_shots].x;
		if (pos[invader_shots].sprite == INVADER_1)
			board[y][x] = sprites[INVADER_1_MISSLE].graph;
		else if (pos[invader_shots].sprite == INVADER_2)
			board[y][x] = sprites[INVADER_2_MISSLE].graph;
		else if (pos[invader_shots].sprite == INVADER_3)
			board[y][x] = sprites[INVADER_3_MISSLE].graph;
	}

	ctrl.invaders_shot_inprogress++;
}

/*
 * RETURN codes:
 *	1 = Invader destroys player.
 */
int
board_update_invaders_missle(void)
{
	int r = 0;
	int y, x, sprite, sprite_next;
	int defense_damage;

	board_backing_store(BBS_BACKUP);
	board_backing_store(BBS_REMOVE_INVADERS_MISSLE);

	for (y = 3; y < (BOARD_ROWS-2); y++) {
		for (x = 0; x < BOARD_COLS; x++) {
			sprite = lookup_sprite(board[y][x]);
			if (sprite != INVADER_1_MISSLE &&
			    sprite != INVADER_2_MISSLE &&
			    sprite != INVADER_3_MISSLE)
				continue;

			if (y == 21) {
				/* Invader hits nothing. */
				board_bs[y][x] = sprites[SPACE].graph;
				ctrl.invaders_shot_inprogress--;
				continue;
			}

			sprite_next = lookup_sprite(board[y+1][x]);

			if (sprite_next == SPACE) {
				/* The way is free. */
				board_bs[y+1][x] = sprites[sprite].graph;
			}

			if (sprite_next == PLAYER_LASER) {
				/* Invader shot passes player shot. */
				DM("Invader shot passes player shot.");
				board_bs[y][x] = sprites[sprite_next].graph;
				board_bs[y+1][x] = sprites[sprite].graph;
			}

			if (sprite_next == SHIELD_3 ||
			    sprite_next == SHIELD_2 ||
			    sprite_next == SHIELD_1) {
				/* Invader damages player defense. */
				defense_damage = 
				    sprites[sprite_next].protection -
				    sprites[sprite].damage;
				switch (defense_damage) {
				case 2:
					/* Player defense damaged. */
					board_bs[y+1][x] =
					    sprites[SHIELD_2].graph;
					break;
				case 1:
					/* Player defense weak. */
					board_bs[y+1][x] =
					    sprites[SHIELD_1].graph;
					break;
				default:
					/* Player defense cracked. */
					board_bs[y+1][x] =
					    sprites[SPACE].graph;
					break;
				}
				board_bs[y][x] = sprites[SPACE].graph;
				ctrl.invaders_shot_inprogress--;
			}

			if (sprite_next == PLAYER_LASER_BASE) {
				/* Invader destroys player. */
				DM("Invader destroys player.");
				ctrl.invaders_shot_inprogress--;
				board_bs[y+1][x] = sprites[EXPLOSION].graph;
				r = 1;
			}
		}
	}

	board_backing_store(BBS_RESTORE);

	return r;
}

void
board_update_mystery_move(void)
{
	int x, sprite, sprite_next;

	for (x = 0; x < BOARD_COLS; x++) {
		sprite = lookup_sprite(board[3][x]);
		if (sprite != MYSTERY_SHIP)
			continue;

		if ((x-1) == -1) {
			/* Mystery ship touches left border. */
			board[3][x] = sprites[SPACE].graph;
			ctrl.mystery_inprogress--;
			DM("Mystery ship leaves screen.");
			break;
		}

		sprite_next = lookup_sprite(board[3][x-1]);

		if (sprite_next == PLAYER_LASER) {
			/* Playershot gets eliminated. */
			ctrl.player_shot_inprogress--;
		}

		board[3][x-1] = sprites[sprite].graph;
		board[3][x] = sprites[SPACE].graph;
	}
}

void
board_update_mystery_trigger(void)
{
	if (ctrl.mystery_inprogress) {
		/* Mystery ship already entered. */
		return;
	}

	if (arc4random_uniform(32) == 4) {
		/* Mystery ship enters screen. */
		DM("Mystery ship enters screen.");
		ctrl.mystery_inprogress++;
		board[3][79] = sprites[MYSTERY_SHIP].graph;
	}

}

void
board_backing_store(int action)
{
	int y, x, sprite;

	if (action == BBS_BACKUP) {
		for (y = 0; y < BOARD_ROWS; y++)
			memcpy(board_bs[y], board[y], BOARD_COLS);
	}

	if (action == BBS_RESTORE) {
		for (y = 0; y < BOARD_ROWS; y++)
			memcpy(board[y], board_bs[y], BOARD_COLS);
	}

	if (action == BBS_REMOVE_INVADERS) {
		for (y = 3; y < (BOARD_ROWS-2); y++) {
			for (x = 0; x < BOARD_COLS; x++) {
				sprite = lookup_sprite(board_bs[y][x]);
				if (sprite == INVADER_1 ||
				    sprite == INVADER_2 ||
				    sprite == INVADER_3)
					board_bs[y][x] = sprites[SPACE].graph;
			}
		}
	}

	if (action == BBS_REMOVE_INVADERS_MISSLE) {
		for (y = 3; y < (BOARD_ROWS-2); y++) {
			for (x = 0; x < BOARD_COLS; x++) {
				sprite = lookup_sprite(board_bs[y][x]);
				if (sprite == INVADER_1_MISSLE ||
				    sprite == INVADER_2_MISSLE ||
				    sprite == INVADER_3_MISSLE)
					board_bs[y][x] = sprites[SPACE].graph;
			}
		}
	}
}

/*
 * RETURN codes:
 *	-1 = Sprite not found.
 *	<sprite> = Successful.
 */
int
lookup_sprite(char graph)
{
	int i;

	for (i = 0; i < SPRITES_NUM; i++) {
		if (graph == sprites[i].graph)
			return i;
	}

	return -1;
}

void
player_score(int sprite)
{
	int n, score;

	beep();

	/* Update score. */
	if (sprite == MYSTERY_SHIP) {
		/* Mystery ship gets random points of 50, 100, or 150. */
		while ((n = arc4random_uniform(4)) == 0);
		score = (sprites[sprite].score * n);
		board_write(0, 35, 1, "%s %d", text_mystery, score);
	} else {
		score = sprites[sprite].score;
		ctrl.invaders_num--;
		/* Make invaders faster.
		 *
		 * History comment (source Wikipedia):
		 * While programming the game, Nishikado discovered that the
		 * processor was able to render the alien graphics faster the
		 * fewer were on screen.  Rather than design the game to
		 * compensate for the speed increase, he decided to keep it
		 * as a challenging gameplay mechanism.
		 */
		if (ctrl.invaders_move_usec > INVADERS_MOVE_MIN_USEC)
			ctrl.invaders_move_usec -= 15000;
	}
	if ((ctrl.player_score + score) <= SCORE_MAX) {
		ctrl.player_score += score;
		board_write(1, 0, 0, "%04d", ctrl.player_score);
	}
	if (ctrl.player_score >= ctrl.player_hiscore) {
		ctrl.player_hiscore = ctrl.player_score;
		board_write(1, 16, 0, "%04d", ctrl.player_hiscore);
		if (ctrl.player_hiscore_new == 0) {
			/* Tell player about new hiscore reached. */
			ctrl.player_hiscore_new++;
			board_write(0, 35, 1, "%s", text_hiscore_new);
		}
	}

	if (ctrl.player_score >= ctrl.score_extra_live) {
		/* Player gets one extra live when certain score is reached. */
		ctrl.player_lives++;
		ctrl.score_extra_live = SCORE_MAX+1;
		board_write(0, 35, 1, "%s", text_extra_live);
		player_redraw_lives();
	}

	if (ctrl.invaders_num == 0) {
		/* Player wins game. */
		timer_clear();
		board_write(0, 35, 0, "%s", text_player_wins);
		board_draw();
		sleep(PAUSE_SEC);
		game_reset(1);
	}
}

void
player_destroy(void)
{
	beep();
	timer_clear();

	/* Decrease player lives. */
	ctrl.player_lives--;
	player_redraw_lives();

	if (ctrl.player_lives == 0) {
		/* Player has no more lives. */
		game_over();
		return;
	}

	/* Reset player base to start position. */
	board[21][ctrl.player_pos_x] = sprites[SPACE].graph;
	ctrl.player_pos_x = 14;
	board[21][ctrl.player_pos_x] = sprites[PLAYER_LASER_BASE].graph;

	/* Have a break before restart. */
	sleep(PAUSE_SEC);
	flush_stdin();
	timer_setup();
}

void
player_redraw_lives(void)
{
	int y, x;

	for (x = 0; x < BOARD_COLS; x++)
		board[23][x] = sprites[SPACE].graph;

	board_write(23, 0, 0, "%d", ctrl.player_lives);
	for (y = 0, x = 8; y < (ctrl.player_lives - 1); y++, x += 2)
		board[23][x] = sprites[PLAYER_LASER_BASE].graph;
}

void
game_reset(int mode)
{
	flush_stdin();
	ctrl_init(mode);
	board_init(0);
	timer_setup();
}

void
game_pause(void)
{
	if (ctrl.game_pause == 0) {
		timer_clear();
		ctrl.game_pause = 1;
		board_write(0, 35, 0, "%s", text_game_pause);
		board_draw();
	} else {
		ctrl.game_pause = 0;
		memset(board[0]+35, sprites[SPACE].graph,
		    strlen(text_game_pause));
		timer_setup();
	}
}

void
game_over(void)
{
	timer_clear();
	ctrl.game_over = 1;
	board_write(0, 35, 0, "%s", text_game_over);
	board_draw();
}

/*
 * RETURN codes:
 *	0 = Some error happened.
 *	<character> = Successful.
 */
char
getkey(void)
{
	int nfds;
	char ch;
	struct pollfd fdpoll[1];

	fdpoll[0].fd = STDIN_FILENO;
	fdpoll[0].events = POLLIN;

	while (1) {
		nfds = poll(fdpoll, 1, -1);
		if (nfds == -1) {
			if (errno == EINTR)
				continue;
			break;
		}
		if (nfds == 0)
			continue;
		if (fdpoll[0].revents & POLLIN) {
			read(STDIN_FILENO, &ch, 1);
			return ch;
		}
	}

	return 0;
}

void
flush_stdin(void)
{
	char ch;

	while (read(STDIN_FILENO, &ch, 1) != -1);
}

void
debug_setup(void)
{
	debug_fd = open(FILENAME_DEBUG, O_CREAT|O_APPEND|O_WRONLY, 0644);
	if (debug_fd == -1)
		err(1, "open");
}

void
debug_write(char *msg, ...)
{
	char buf[1024];
	va_list ap;

	if (!debug_fd)
		return;

	va_start(ap, msg);
	vsnprintf(buf, sizeof(buf), msg, ap);
	va_end(ap);

	write(debug_fd, buf, strlen(msg));
	write(debug_fd, "\n", 1);
}
